<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Zomer
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function zomer_body_classes( $classes ) {
	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Checks if current page is not singular.
	if ( ! is_singular() ) {
        // Adds a class if blog does not display excerpts.
        $classes[] = 'archive-masonry';

		// Adds a class of hfeed to non-singular pages.
		$classes[] = 'hfeed';
	}

	// Adds a class if a featured image is not set in single views
	if ( is_singular() && has_post_thumbnail() ) {
		$classes[] = 'has-featured-image';
	}

	// Adds a class if a sidebar is not active.
	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$classes[] = 'inactive-sidebar';
	}

	// Adds a class if a social menu is not set.
	if ( ! has_nav_menu( 'social' ) ) {
		$classes[] = 'inactive-social-menu';
	}

	return $classes;
}
add_filter( 'body_class', 'zomer_body_classes' );

/**
 * Prints featured image URL with a custom image size.
 */
function zomer_featured_image_url() {
	$thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( get_the_ID() ), 'zomer-fullscreen-image' );

	return $thumbnail[0];
}

/**
 * Returns a class for the #primary div based on sidebar visibility
 */
function zomer_get_blog_primary_class() {
    $primary_class = 'col-lg-8 col-md-8';

	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$primary_class = 'col-lg-12';
	}

    return apply_filters( 'zomer_blog_primary_class', $primary_class );
}

/**
 * Appends the wp_link_pages to the content before share buttons.
 */
function zomer_add_link_pages_content( $content ) {
	if ( is_admin() ) {
		return $content;
	}

	if ( in_the_loop() ) {
		$args = array(
			'before'      => '<div class="page-links"><span class="page-links-title">' . esc_html__( 'Pages:', 'zomer' ) . '</span>',
			'after'       => '</div>',
			'link_before' => '<span>',
			'link_after'  => '</span>',
			'pagelink'    => '<span class="screen-reader-text">' . esc_html__( 'Page', 'zomer' ) . ' </span>%',
			'separator'   => '<span class="screen-reader-text">, </span>',
			'echo'        => false,
			'_show'       => true,
		);

		$content .= wp_link_pages( $args );
	}

	return $content;
}
add_filter( 'the_content', 'zomer_add_link_pages_content', 10 );
