<?php
/**
 * Zomer functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Zomer
 */

if ( ! function_exists( 'zomer_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function zomer_setup() {
		/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on Zomer, use a find and replace
		* to change 'zomer' to the name of your theme in all the template files.
		*/
		load_theme_textdomain( 'zomer', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
		add_theme_support( 'title-tag' );

		/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'zomer-fullscreen-image', 1440, 9999 );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' => esc_html__( 'Header Menu', 'zomer' ),
			'social' => esc_html__( 'Social Menu', 'zomer' ),
		) );

		// Enable support for custom logo.
		add_theme_support( 'custom-logo', array(
			'height'      => 340,
			'width'       => 340,
			'flex-height' => true,
			'header-text' => array( 'site-title' ),
		) );

		/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'zomer_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		/*
		* This theme styles the visual editor to resemble the theme style.
		*/
		add_editor_style( array( 'css/editor-style.css', zomer_google_fonts() ) );
	}
endif;
add_action( 'after_setup_theme', 'zomer_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function zomer_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'zomer_content_width', 765 );
}
add_action( 'after_setup_theme', 'zomer_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function zomer_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'zomer' ),
		'description'   => esc_html__( 'Appears in the sidebar section of the site.', 'zomer' ),
		'id'            => 'sidebar-1',
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer', 'zomer' ),
		'description'   => esc_html__( 'Appears in the footer section of the site.', 'zomer' ),
		'id'            => 'sidebar-2',
		'before_widget' => '<div class="footer-widget"><aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside></div><!-- .footer-widget -->',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
}
add_action( 'widgets_init', 'zomer_widgets_init' );

/**
 * Register Google Fonts
 */
function zomer_google_fonts() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext,cyrillic,cyrillic-ext';

	/* translators: If there are characters in your language that are not supported by Noto Serif, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Noto Serif font: on or off', 'zomer' ) ) {
		$fonts[] = 'Noto Serif:400,400i,700,700i';
	}

	/* translators: If there are characters in your language that are not supported by Noto Sans, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Noto Sans font: on or off', 'zomer' ) ) {
		$fonts[] = 'Noto Sans:400,400i,700,700i';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}

/**
 * Enqueue scripts and styles.
 */
function zomer_scripts() {
	wp_enqueue_style( 'zomer-google-fonts', zomer_google_fonts(), array(), null );
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() . '/css/font-awesome.css' );
	wp_enqueue_style( 'zomer-style', get_stylesheet_uri() );

	wp_enqueue_script( 'zomer-navigation', get_template_directory_uri() . '/js/navigation.js', array(), '20151215', true );

	wp_enqueue_script( 'zomer-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20190801', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	if ( ! is_singular() ) {
		wp_enqueue_script( 'masonry' );
	}

	wp_register_script( 'jquery-big-slide', get_template_directory_uri() . '/js/big-slide.js', array( 'jquery' ), '0.9.3', true );
	wp_enqueue_script( 'zomer-script', get_template_directory_uri() . '/js/zomer.js', array( 'jquery-big-slide' ), '1.0.0', true  );
}
add_action( 'wp_enqueue_scripts', 'zomer_scripts' );

/**
 * Modifies tag cloud widget arguments to have all tags in the widget same font size.
 */
function zomer_widget_tag_cloud_args( $args ) {
		$args['largest'] = 0.875;
		$args['smallest'] = 0.875;
		$args['unit'] = 'em';
		return $args;
}
add_filter( 'widget_tag_cloud_args', 'zomer_widget_tag_cloud_args' );

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}

/**
 * Load Dashboard help file.
 */
require get_template_directory() . '/inc/help/helper.php';
