<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features
 *
 * @package Latest
 */

/**
 * Adds custom classes to the array of body classes
 *
 * @param array $classes Classes for the body element
 * @return array
 */
function latest_body_classes( $classes ) {
	// Adds a class of group-blog to blogs with more than 1 published author
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	if ( get_theme_mod( 'header_textcolor' ) == 'blank' ) {
		$classes[] = 'title-tagline-hidden';
	}

	if ( post_password_required() ) {
		$classes[] = 'post-password-required';
	}

	return $classes;
}
add_filter( 'body_class', 'latest_body_classes' );


/**
 * Adds post class(es) for posts in blog index, archives and search results
 */
function latest_post_classes( $classes ) {
	
	if ( is_home() || is_archive() || is_search() ) {
		if ( !get_theme_mod( 'masonry_off' ) ) {
			$classes[] = 'masonry-entry';
			$masonry_columns = get_theme_mod( 'masonry_columns', 3 );
			$classes[] = 'masonry-' . esc_attr( $masonry_columns );
		} else {
			$classes[] = 'non-masonry-entry';
		}
	}

	return $classes;
}
add_filter( 'post_class', 'latest_post_classes' );


function latest_primary_menu_fallback() {
	if ( is_user_logged_in() && current_user_can( 'administrator' ) ) {
		echo '<div><ul id="primary-menu" class="clearfix"><li class="menu-item"><a href="' . esc_url( admin_url( 'nav-menus.php' ) ) . '">' . esc_html__( 'Create your Primary Menu here', 'latest' ) . '</a></li></ul></div>';
	} else {
		return;
	}

}


function latest_custom_excerpt_length( $length ) {
	return 30;
}
add_filter( 'excerpt_length', 'latest_custom_excerpt_length', 999 );


function latest_excerpt_more( $more ) {
    return sprintf( '&hellip; <a class="read-more" href="%1$s" title="%2$s">%3$s</a>',
        esc_url( get_permalink( get_the_ID() ) ),
        esc_html( get_the_title( get_the_ID() ) ),
        '<i class="fa fa-arrow-right"></i>'
    );
}
add_filter( 'excerpt_more', 'latest_excerpt_more' );


function latest_css_font_family($font_family) {
	$font_family = substr($font_family, 0, strpos($font_family, ':' ));
	return esc_attr($font_family);
}


function latest_dynamic_style( $css = array() ){

	$sidebar_position = get_theme_mod( 'sidebar_position' );
	if ( $sidebar_position == "left" ) {
		$css[] = '#primary{float:right;}#secondary{float:left;}';
	}

	$header_textcolor = get_theme_mod( 'header_textcolor', '2d3139' );
	if ( $header_textcolor && $header_textcolor != '2d3139' && $header_textcolor != 'blank' ) {
		$css[] = '.site-title a,.site-description{color:#' . esc_attr($header_textcolor) . ';}';
	}

	$header_bg = get_header_image();
	if ( $header_bg ) {
		$css[] = '#masthead{background-image: '.'url( ' . $header_bg . ' )}';
	}

	$hi_color = get_theme_mod( 'hi_color', '#ff5a79' );
	if ( $hi_color && $hi_color != '#ff5a79' ) {
		$hi_color = esc_attr($hi_color);
		$css[] = '.button,a.button,button,input[type="button"],input[type="reset"],input[type="submit"],.related-posts h2:before,#top-bar a.button,#top-bar button,#top-bar input[type="button"],#top-bar input[type="reset"],#top-bar input[type="submit"],#colophon h5.widget-title:after{background:' . $hi_color . ';}';
		$css[] = '.mejs-overlay-button,.home .entry-video .mejs-overlay-button,.blog .entry-video .mejs-overlay-button,.archive .entry-video .mejs-overlay-button,.search .entry-video .mejs-overlay-button{background-color:' . $hi_color . ';}';
		$css[] = 'a,a:hover,a:focus,a:active,.single-entry-content a,.latest-entry-icon:before,.latest-entry-icon:after,.entry-header .entry-title a:hover,.entry-header.with-image .entry-title a:hover,.entry-header .entry-title .entry-cat,.entry-header .entry-title .entry-cat a,.entry-header.with-image .entry-title .entry-cat,.entry-header.with-image .entry-title .entry-cat a,.entry-header.with-image .entry-title .entry-cat-sep,.entry-header.with-image .entry-meta .entry-meta-sep,.entry-meta span .fa,.entry-footer span .fa,.entry-footer span.tags-links,.comment-list a:hover,.comment-navigation .nav-previous a,.comment-navigation .nav-next a,#top-bar a,#top-bar a:hover,#top-bar .fa,#primary-menu li a:hover,#primary-menu li.current-menu-item a,#primary-menu li:hover.menu-item-has-children:after,.single #primary-menu li.primary-category a,#primary-menu ul li a:hover,.pagination a:hover,.pagination .current{color:' . $hi_color . ';}';
		$css[] = '.sticky,#primary-menu > li:hover,#primary-menu > li.current-menu-item,.single #primary-menu > li.primary-category{border-color:' . $hi_color . ';}';
		$css[] = '.comment-navigation .nav-next a:after{border-left-color:' . $hi_color . ';}';
		$css[] = '.comment-navigation .nav-previous a:after{border-right-color:' . $hi_color . ';}';
		$css[] = '#footer-menu a[href*="codepen.io"]:before,#footer-menu a[href*="digg.com"]:before,#footer-menu a[href*="dribbble.com"]:before,#footer-menu a[href*="dropbox.com"]:before,#footer-menu a[href*="facebook.com"]:before,#footer-menu a[href*="flickr.com"]:before,#footer-menu a[href*="foursquare.com"]:before,#footer-menu a[href*="plus.google.com"]:before,#footer-menu a[href*="github.com"]:before,#footer-menu a[href*="instagram.com"]:before,#footer-menu a[href*="linkedin.com"]:before,#footer-menu a[href*="pinterest.com"]:before,#footer-menu a[href*="getpocket.com"]:before,#footer-menu a[href*="reddit.com"]:before,#footer-menu a[href*="skype.com"]:before,#footer-menu a[href*="stumbleupon.com"]:before,#footer-menu a[href*="tumblr.com"]:before,#footer-menu a[href*="twitter.com"]:before,#footer-menu a[href*="vimeo.com"]:before,#footer-menu a[href*="wordpress.com"]:before,#footer-menu a[href*="wordpress.org"]:before,#footer-menu a[href*="youtube.com"]:before,#footer-menu a[href^="mailto:"]:before,#footer-menu a[href*="spotify.com"]:before,#footer-menu a[href*="twitch.tv"]:before,#footer-menu a[href$="/feed/"]:before{box-shadow: inset 0 0 0 ' . $hi_color . ';}';
		$css[] = '#footer-menu a[href*="codepen.io"]:hover:before,#footer-menu a[href*="digg.com"]:hover:before,#footer-menu a[href*="dribbble.com"]:hover:before,#footer-menu a[href*="dropbox.com"]:hover:before,#footer-menu a[href*="facebook.com"]:hover:before,#footer-menu a[href*="flickr.com"]:hover:before,#footer-menu a[href*="foursquare.com"]:hover:before,#footer-menu a[href*="plus.google.com"]:hover:before,#footer-menu a[href*="github.com"]:hover:before,#footer-menu a[href*="instagram.com"]:hover:before,#footer-menu a[href*="linkedin.com"]:hover:before,#footer-menu a[href*="pinterest.com"]:hover:before,#footer-menu a[href*="getpocket.com"]:hover:before,#footer-menu a[href*="reddit.com"]:hover:before,#footer-menu a[href*="skype.com"]:hover:before,#footer-menu a[href*="stumbleupon.com"]:hover:before,#footer-menu a[href*="tumblr.com"]:hover:before,#footer-menu a[href*="twitter.com"]:hover:before,#footer-menu a[href*="vimeo.com"]:hover:before,#footer-menu a[href*="wordpress.com"]:hover:before,#footer-menu a[href*="wordpress.org"]:hover:before,#footer-menu a[href*="youtube.com"]:hover:before,#footer-menu a[href^="mailto:"]:hover:before,#footer-menu a[href*="spotify.com"]:hover:before,#footer-menu a[href*="twitch.tv"]:hover:before,#footer-menu a[href$="/feed/"]:hover:before{background:' . $hi_color . ';box-shadow: inset 0 -40px 0 ' . $hi_color . ';}';
	}

	return implode( '', $css );

}


function latest_editor_dynamic_style( $mceInit, $css = array() ) {

	$hi_color = get_theme_mod( 'hi_color' );
	if ($hi_color && $hi_color != "#ff5a79") {
		$css[] = '.mce-content-body a,.mce-content-body a:hover,.mce-content-body a:focus,.mce-content-body a:active{color:'.esc_attr($hi_color).'}';
	}

	$styles = implode( '', $css );

    if ( isset( $mceInit['content_style'] ) ) {
        $mceInit['content_style'] .= ' ' . $styles . ' ';
    } else {
        $mceInit['content_style'] = $styles . ' ';
    }
    return $mceInit;

}
add_filter( 'tiny_mce_before_init', 'latest_editor_dynamic_style' );


function latest_block_editor_dynamic_style( $css = array() ) {

	$hi_color = get_theme_mod( 'hi_color' );
	if ($hi_color && $hi_color != "#ff5a79") {		
		$css[] = '.editor-rich-text__tinymce a,.editor-rich-text__tinymce a:hover,.editor-rich-text__tinymce a:focus,.editor-rich-text__tinymce a:active{color:'.esc_attr($hi_color).'}';
	}

    return implode( '', $css );

}


/**
 * Get category list, excluding primary category
 */
if(!function_exists( 'latest_get_the_category_list' )){
	function latest_get_the_category_list($exclude,$separator = false) {

		if ( !$separator ) {
			$separator = ' / ';
		}
		$categories = get_the_category();

		$output = '';
		if ( $categories ) {
			foreach ( $categories as $category ) {
				if( $category->term_id != $exclude ) {
					$output .= '<a href="' . get_category_link( $category->term_id ) . '">' . $category->cat_name . '</a>' . $separator;
				}
			}
			$output = trim( $output, $separator );
		}

		return $output;
	}
}


/**
 * Page/post/archive title with featured image background or video content
 */
if(!function_exists( 'latest_header_title' )){
	function latest_header_title() { 
		?>

<?php if ( is_singular() && !is_front_page() ) {

	if ( get_post_format() == 'video' ) {
		$content = apply_filters( 'the_content', get_post( get_the_ID() )->post_content );
		$video = false;
		// Only get video from the content if a playlist isn't present.
		if ( false === strpos( $content, 'wp-playlist-script' ) ) {
			$video = get_media_embedded_in_content( $content, array( 'video', 'object', 'embed', 'iframe' ) );
		}
		if ( ! empty( $video ) ) {
			$video_entry = 'on';
			$first_video = true;
			foreach ( $video as $video_html ) {
				if ( $first_video ) {
					echo '<div class="container video-container">
					<div id="entry-video" class="entry-video single">';
						echo $video_html;
					echo '</div>
					</div>';
					$first_video = false;
				}
			}
		} else {
			$video_entry = 'off';
		};

	} else {
	$video_entry = 'off';
	}

	if ( has_post_thumbnail() && $video_entry == 'off' ) { 
		$image_attributes = wp_get_attachment_image_src( get_post_thumbnail_id(), 'full' );
		if ( $image_attributes[1] > $image_attributes[2] ) { ?>
        	<header class="container entry-header singular with-image">
        		<div class="title-wrapper" style="background-image: url('<?php echo esc_url( $image_attributes[0] );?>')">
		<?php } else { ?>
			<header class="container entry-header singular">
				<div class="title-wrapper">	
		<?php }
		} else { ?>
		<header class="container entry-header singular">
			<div class="title-wrapper">
	<?php } ?>
			<?php the_title( '<h1 class="entry-title"><i class="latest-entry-icon"></i>', '</h1>' ); ?>
			</div>
		</header><!-- .entry-header.singular -->
	<?php
		$latest_excerpt = apply_filters( 'the_excerpt', get_post_field( 'post_excerpt') );
		if ( $latest_excerpt ) {
			echo '<div class="entry-excerpt container">';
			echo $latest_excerpt;
			echo '</div>';
		}
	if ( is_single() && is_active_sidebar( 'latest-sidebar' ) ) { ?>
	<div class="entry-meta container">
		<?php $author_id = get_post_field( 'post_author', get_the_ID() );
			$author_name = get_the_author_meta( 'display_name', $author_id );
			$author_url = get_author_posts_url( $author_id );
			$author_avatar = get_avatar( $author_id, 80);
			echo '<div class="byline"><span class="author vcard"><a class="url fn n" href="' . esc_url( $author_url ) . '">' . $author_avatar . esc_html( $author_name ) . '</a></div>';
			if ( function_exists( 'epc_get_primary_term_posts' ) ) {
				$primary_category = get_post_meta( get_the_ID(), 'epc_primary_category', true );
				if ( $primary_category ) {
					$the_category = '<a href="' . esc_url( get_category_link( $primary_category ) ) . '" class="entry-cat">' . esc_html( get_cat_name( $primary_category ) ) . '</a><span class="entry-cat-sep"> / </span>';
				} else {
					$categories = get_the_category();
					if ( ! empty( $categories ) ) {
						$the_category = '<a href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '" class="entry-cat">' . esc_html( $categories[0]->name ) . '</a><span class="entry-cat-sep"> / </span>';
					} else {
						$the_category = '';
					}					
				}
				echo $the_category;
			}
		latest_posted_on(); ?>
	</div><!-- .entry-meta -->
	<?php } ?>
<?php } elseif ( is_archive() ) { ?>
		<header class="container entry-header archive">
			<div class="entry-wrapper">
			<?php if ( is_tag() ) { ?>
				<h1 class="entry-title"><span class="hashtag">#</span><?php echo single_cat_title(); ?></h1>
			<?php } elseif ( is_category() ) { ?>
				<h1 class="entry-title"><?php echo single_cat_title(); ?></h1>
			<?php } else {
				the_archive_title( '<h1 class="entry-title">', '</h1>' );
				}
				the_archive_description( '<div class="taxonomy-description">', '</div>' );
				?>
			</div>
		</header><!-- .entry-header.archive -->	
	<?php } elseif ( is_search() ) { ?>
		<header class="container entry-header search">
			<div class="entry-wrapper">
			<h1 class="entry-title"><?php printf( esc_html__( 'Search Results for: %s', 'latest' ), '<i>' . get_search_query() . '</i>' ); ?></h1>
			</div>
		</header><!-- .entry-header.search -->	
	<?php }	elseif ( is_404() ) { ?>
		<header class="container entry-header not-found">
			<div class="entry-wrapper">
			<h1 class="entry-title"><?php esc_html_e( '404 Error - Not Found', 'latest' ); ?></h1>
			</div>
		</header><!-- .entry-header.not-found -->	
	<?php } ?>

	<div class="container clearfix">
	<?php }
}


/**
 * Remove first embedded media (video) in post content,
 * as we already output first video in head of post
 */
if(!function_exists( 'latest_strip_media_embedded_in_content' )){
	function latest_strip_media_embedded_in_content($content){

		$content = preg_replace("#<(?P<tag>video)[^<]*?(?:>[\s\S]*?<\/(?P=tag)>|\s*\/>)#", "", $content, 1);
		$content = preg_replace("#<(?P<tag>object)[^<]*?(?:>[\s\S]*?<\/(?P=tag)>|\s*\/>)#", "", $content, 1);
		$content = preg_replace("#<(?P<tag>embed)[^<]*?(?:>[\s\S]*?<\/(?P=tag)>|\s*\/>)#", "", $content, 1);
		$content = preg_replace("#<(?P<tag>iframe)[^<]*?(?:>[\s\S]*?<\/(?P=tag)>|\s*\/>)#", "", $content, 1);

		return $content;

	}
}


/**
 * Powered by WordPress
 */
if(!function_exists( 'latest_powered_by' )){
	function latest_powered_by(){
		?>
				<div class="site-info">
				<?php if ( function_exists( 'the_privacy_policy_link' ) && !latest_privacy_in_menu() ) {
					the_privacy_policy_link( '', '<span class="sep privacy"> | </span>' );
				} ?>
					<a href="<?php echo esc_url( __( 'https://wordpress.org/', 'latest' ) ); ?>"><?php printf( esc_html__( 'Powered by %s', 'latest' ), 'WordPress' ); ?></a>
					<span class="sep"> | </span>
					<?php printf( esc_html__( 'Theme: %2$s by %1$s', 'latest' ), 'uXL Themes', '<a href="https://uxlthemes.com/theme/latest/" rel="designer">Latest</a>' ); ?>
				</div>
		<?php
	}
}


/**
 * Check if Privacy Policy page is already in theme primary or footer menu
 */
if( ! function_exists( 'latest_privacy_in_menu' ) ) {
	function latest_privacy_in_menu(){

		$menu_page_ids = array();

		$locations = get_nav_menu_locations();

		if ( has_nav_menu( 'primary' ) ) {
			$menu_name_primary = wp_get_nav_menu_object( $locations[ 'primary' ] );
			$menu_items_primary = wp_get_nav_menu_items( $menu_name_primary );
			foreach ( $menu_items_primary as $menu_item_primary ) {
				if ( $menu_item_primary->object == 'page' ) {
					$menu_page_ids[] = $menu_item_primary->object_id;
				}
			}
		}

		if ( has_nav_menu( 'footer' ) ) {
			$menu_name_footer = wp_get_nav_menu_object( $locations[ 'footer' ] );
			$menu_items_footer = wp_get_nav_menu_items( $menu_name_footer );
			foreach ( $menu_items_footer as $menu_item_footer ) {
				if ( $menu_item_footer->object == 'page' ) {
					$menu_page_ids[] = $menu_item_footer->object_id;
				}
			}
		}

		$privacy_policy_page_id = (int) get_option( 'wp_page_for_privacy_policy' );
		if ( ! empty( $privacy_policy_page_id ) && get_post_status( $privacy_policy_page_id ) === 'publish' && in_array( $privacy_policy_page_id, $menu_page_ids ) ) {
			return true;
		} else {
			return false;
		}

	}
}